import { useAIAssistantHelpers } from '@/features/ai/assistant/composables/useAIAssistantHelpers';
import { AI_ASSISTANT_MAX_CONTENT_LENGTH } from '@/app/constants';
import type { ICredentialsResponse } from '@/features/credentials/credentials.types';
import type { IRestApiContext } from '@n8n/rest-api-client';
import type {
	AskAiRequest,
	ChatRequest,
	ReplaceCodeRequest,
} from '@/features/ai/assistant/assistant.types';
import { makeRestApiRequest, streamRequest } from '@n8n/rest-api-client';
import { getObjectSizeInKB } from '@/app/utils/objectUtils';
import type { IDataObject } from 'n8n-workflow';

export function chatWithBuilder(
	ctx: IRestApiContext,
	payload: ChatRequest.RequestPayload,
	onMessageUpdated: (data: ChatRequest.ResponsePayload) => void,
	onDone: () => void,
	onError: (e: Error) => void,
	versionId?: string,
	abortSignal?: AbortSignal,
): void {
	void streamRequest<ChatRequest.ResponsePayload>(
		ctx,
		'/ai/build',
		{
			...payload,
			payload: {
				...payload.payload,
				versionId,
			},
		},
		onMessageUpdated,
		onDone,
		onError,
		undefined,
		abortSignal,
	);
}

export function chatWithAssistant(
	ctx: IRestApiContext,
	payload: ChatRequest.RequestPayload,
	onMessageUpdated: (data: ChatRequest.ResponsePayload) => void,
	onDone: () => void,
	onError: (e: Error) => void,
): void {
	try {
		const payloadSize = getObjectSizeInKB(payload.payload);
		if (payloadSize > AI_ASSISTANT_MAX_CONTENT_LENGTH) {
			useAIAssistantHelpers().trimPayloadSize(payload);
		}
	} catch (e) {
		onError(e);
		return;
	}
	void streamRequest<ChatRequest.ResponsePayload>(
		ctx,
		'/ai/chat',
		payload,
		onMessageUpdated,
		onDone,
		onError,
	);
}

export async function replaceCode(
	context: IRestApiContext,
	data: ReplaceCodeRequest.RequestPayload,
): Promise<ReplaceCodeRequest.ResponsePayload> {
	return await makeRestApiRequest<ReplaceCodeRequest.ResponsePayload>(
		context,
		'POST',
		'/ai/chat/apply-suggestion',
		data,
	);
}

export async function generateCodeForPrompt(
	ctx: IRestApiContext,
	{ question, context, forNode }: AskAiRequest.RequestPayload,
): Promise<{ code: string }> {
	const body: IDataObject = {
		question,
		context,
		forNode,
	};
	return await makeRestApiRequest(ctx, 'POST', '/ai/ask-ai', body);
}

export async function claimFreeAiCredits(
	ctx: IRestApiContext,
	{ projectId }: { projectId?: string },
): Promise<ICredentialsResponse> {
	const body: IDataObject = {
		projectId,
	};
	return await makeRestApiRequest(ctx, 'POST', '/ai/free-credits', body);
}

export async function getAiSessions(
	ctx: IRestApiContext,
	workflowId?: string,
): Promise<{
	sessions: Array<{
		sessionId: string;
		messages: ChatRequest.MessageResponse[];
		lastUpdated: string;
	}>;
}> {
	const body: IDataObject = {
		workflowId,
	};
	return await makeRestApiRequest(ctx, 'POST', '/ai/sessions', body);
}

export async function getSessionsMetadata(
	ctx: IRestApiContext,
	workflowId?: string,
): Promise<{
	hasMessages: boolean;
}> {
	const body: IDataObject = {
		workflowId,
	};
	return await makeRestApiRequest(ctx, 'POST', '/ai/sessions/metadata', body);
}

export async function getBuilderCredits(ctx: IRestApiContext): Promise<{
	creditsQuota: number;
	creditsClaimed: number;
}> {
	return await makeRestApiRequest(ctx, 'GET', '/ai/build/credits');
}

export async function truncateBuilderMessages(
	ctx: IRestApiContext,
	workflowId: string,
	messageId: string,
): Promise<{ success: boolean }> {
	return await makeRestApiRequest(ctx, 'POST', '/ai/build/truncate-messages', {
		workflowId,
		messageId,
	});
}
